// TextureDemo.cpp

#include <GL/glut.h>
#include <stdlib.h>
#include <iostream.h>
#include <iomanip.h>
#include <windows.h>
#include <math.h>
#include "Texture.h"

Texture*		texFront;
Texture*		texLeft;
Texture*		texRight;
Texture*		texBack;
Texture*		texTop;
Texture*		texBottom;
GLUquadricObj*  quadricObj;

// global variables controlled by the UI
GLfloat		TexEnd = 1.0;
GLint		rotateY = 0;
GLint		fov = 60;		// field of view
bool		cube = true;

void SetUpTextures()
{
    glEnable(GL_TEXTURE_2D);

	texFront = new Texture();
	texFront->load("angela.bmp");
	if (texFront->getError())
	{
		cerr << texFront->getErrorMessage() << endl;
		exit(1);
	}
	texFront->setParameter(GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	texFront->setParameter(GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	texFront->setParameter(GL_TEXTURE_WRAP_S, GL_REPEAT);

	texBack = new Texture();
	texBack->load("angela.bmp");
	if (texBack->getError())
	{
		cerr << texBack->getErrorMessage() << endl;
		exit(1);
	}
	texBack->setParameter(GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	texBack->setParameter(GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	texBack->setParameter(GL_TEXTURE_WRAP_S, GL_REPEAT);

	texTop = new Texture();
	texTop->load("colors.bmp");
	if (texTop->getError())
	{
		cerr << texTop->getErrorMessage() << endl;
		exit(1);
	}
	texTop->setParameter(GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	texTop->setParameter(GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	texTop->setParameter(GL_TEXTURE_WRAP_S, GL_REPEAT);

	texLeft = new Texture();
	texLeft->load("colors.bmp");
	if (texLeft->getError())
	{
		cerr << texLeft->getErrorMessage() << endl;
		exit(1);
	}
	texLeft->setParameter(GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	texLeft->setParameter(GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	texLeft->setParameter(GL_TEXTURE_WRAP_S, GL_CLAMP);
	texLeft->setParameter(GL_TEXTURE_WRAP_T, GL_CLAMP);

}

void DrawCube()
{
	glNormal3f(0, 0, 1);
	texFront->makeCurrent();
    glBegin(GL_POLYGON);  // front
        glTexCoord2f(0, 0);         glVertex3f(-1, -1,  1);
        glTexCoord2f(0, TexEnd);    glVertex3f( 1, -1,  1);
        glTexCoord2f(1, TexEnd);    glVertex3f( 1,  1,  1);
        glTexCoord2f(1, 0);         glVertex3f(-1,  1,  1);
    glEnd();

	glNormal3f(0, 0, -1);
	texBack->makeCurrent();
    glBegin(GL_POLYGON);  // back
        glTexCoord2f(0, 0);         glVertex3f(-1,  1, -1);
        glTexCoord2f(0, TexEnd);    glVertex3f( 1,  1, -1);
        glTexCoord2f(1, TexEnd);    glVertex3f( 1, -1, -1);
        glTexCoord2f(1, 0);         glVertex3f(-1, -1, -1);
    glEnd();

	glNormal3f(0, 1, 0);
	texTop->makeCurrent();
    glBegin(GL_POLYGON);  // top
        glTexCoord2f(0, 0);         glVertex3f(-1,  1,  1);
        glTexCoord2f(0, TexEnd);    glVertex3f( 1,  1,  1);
        glTexCoord2f(1, TexEnd);    glVertex3f( 1,  1, -1);
        glTexCoord2f(1, 0);         glVertex3f(-1,  1, -1);
    glEnd();

	glNormal3f(0, -1, 0);
	texTop->makeNotCurrent();  // no texture on bottom
    glBegin(GL_POLYGON);  // bottom  (not visible)
        glTexCoord2f(0, 0);         glVertex3f(-1, -1,  1);
        glTexCoord2f(0, TexEnd);    glVertex3f(-1, -1, -1);
        glTexCoord2f(1, TexEnd);    glVertex3f( 1, -1, -1);
        glTexCoord2f(1, 0);         glVertex3f( 1, -1,  1);
    glEnd();

	glNormal3f(-1, 0, 0);
	texLeft->makeCurrent();
    glBegin(GL_POLYGON);  // left
        glTexCoord2f(0, 0);         glVertex3f(-1,  1,  1);
        glTexCoord2f(0, TexEnd);    glVertex3f(-1,  1, -1);
        glTexCoord2f(1, TexEnd);    glVertex3f(-1, -1, -1);
        glTexCoord2f(1, 0);         glVertex3f(-1, -1,  1);
    glEnd();

	glNormal3f(1, 0, 0);
	texLeft->makeNotCurrent();  // no texture
    glBegin(GL_POLYGON);  // right
        glTexCoord2f(0, 0);         glVertex3f( 1,  1,  1);
        glTexCoord2f(0, TexEnd);    glVertex3f( 1, -1,  1);
        glTexCoord2f(1, TexEnd);    glVertex3f( 1, -1, -1);
        glTexCoord2f(1, 0);         glVertex3f( 1,  1, -1);
    glEnd();

}

void SetFieldOfView()
{
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(fov, 1.0, 0.5, 200);
    glMatrixMode(GL_MODELVIEW);	// stays modelview for rest of pgm
}

void HandleReshape(int w, int h)
{
	SetFieldOfView();
}

void HandleDisplay(void)
{
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();
	gluLookAt(0, 3, 5,   0, 0, 0,   0, 1, 0);

    glColor3f(1, 1, 1);
	glRotatef(rotateY, 0, 1, 0);

	if (cube)
		DrawCube();
	else
	{
		texFront->makeCurrent();
		gluSphere(quadricObj, 1.0, 10, 10);  // radius, slices, stacks
	}
	
	glutSwapBuffers();
}

void PrintHelp(void)
{
    cout << "Options:\n";
    cout << "?, h, H:              This help.\n";
    cout << "q, Q, ESC:            Exit program.\n";
	cout << "left, right arrows    Rotate cube\n";
	cout << "e                     Toggle TexEnd\n";
	cout << "c                     Toggle cube / sphere\n";
	cout << "space                 Reset\n";
	cout << endl;
}

void HandleKeyboard(unsigned char key, int x, int y)
{
    switch (key)
    {
    case 'q':
    case 'Q':
    case 27:    // Escape
        exit(0);
        break;
    case 'h':
    case 'H':
    case '?':
        PrintHelp();
        break;
	case ' ':
		rotateY = 0;
		cube = true;
		TexEnd = 1;
		fov = 60;
		break;
	case 'c':
		cube = !cube;
		break;
    case 'e':
        TexEnd /= 2.0;   // cycle 1.0, 0.5, 0.25, 8.0, 4.0, 2.0, 1.0, ...
        if (TexEnd < 0.25f)
            TexEnd = 8.0;
        cout << "TexEnd = " << TexEnd << endl;
        break;
    default:
        cout << "Unrecognized character: " << 
                            key << " " << (int)key << endl;
        return;
    }
    glutPostRedisplay();
}
	
void HandleSpecial(int key, int x, int y)
{
    switch (key)
    {
    case GLUT_KEY_PAGE_UP:
        if (fov < 175)
			fov += 5;
		SetFieldOfView();
        break;
    case GLUT_KEY_PAGE_DOWN:
        if (fov > 5)
			fov -= 5;
		SetFieldOfView();
        break;
    case GLUT_KEY_LEFT:
        rotateY = (rotateY + 345) % 360;
        break;
    case GLUT_KEY_RIGHT:
        rotateY = (rotateY + 15) % 360;
        break;
    default:
        cout << "Unrecognized special character:" << key << endl;
    }
    glutPostRedisplay();
}

int main(int argc, char** argv)
{
    glutInit(&argc, argv);
    glutInitDisplayMode(GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH);
    glutInitWindowSize(500, 500);
    glutInitWindowPosition(100, 100);
    glutCreateWindow("Texture Demo Program");
    glutDisplayFunc(HandleDisplay);
    glutReshapeFunc(HandleReshape);
    glutKeyboardFunc(HandleKeyboard);
    glutSpecialFunc(HandleSpecial);
    glClearColor(0.5, 0.5, 0.8, 1.0);
    glEnable(GL_CULL_FACE);     // back-face culling
    glCullFace(GL_BACK);
    glEnable(GL_DEPTH_TEST);    // z-buffering

    glViewport(0, 0, 500, 500);

	SetUpTextures();

    // Create a quadric objects so we can have a sphere with
    // texture coordinates
    quadricObj = gluNewQuadric();
    if (quadricObj == 0)
        cout << "***gluNewQuadric failed.***\n";
    gluQuadricTexture(quadricObj, GL_TRUE);  // please generate tex coords
    // this quadric object should be deleted with gluDeleteQuadric()
	// but there's no convenient place to do that in a glut program


	cout << "Version: " << glGetString(GL_VERSION)
		 << " Vendor: " << glGetString(GL_VENDOR)
		 << " Renderer: " << glGetString(GL_RENDERER) << endl;

    glutMainLoop();
    return 0;
}
