// Texture.cpp

#include <stdlib.h>
#include <stdio.h>
#include "Texture.h"

Texture::Texture()
{
	error = false;
	textureErrorMessage = new char[200];
	ParmWrapS = ParmWrapT = ParmMagFilter = ParmMinFilter = 0;
}

void Texture::load(char* filename)
{
	// if (filename ends in .bmp)
	loadFromBMP(filename);
	if (error)
		return;

	// Ask OpenGL for a GLuint "name" to identify this texture
	glGenTextures (1, &textureName);
	
	// Create a texture object
	glBindTexture(GL_TEXTURE_2D, textureName);
    GLenum errCode;
	errCode = glGetError();
	if (errCode != GL_NO_ERROR)
	{
		sprintf(textureErrorMessage, 
				"glBindTexture Error: %s\n", gluErrorString(errCode));
		error = true;
		return;
	}

	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, width, height, 0, 
				 GL_RGBA, GL_UNSIGNED_BYTE, textureBytes);

	errCode = glGetError();
	if (errCode != GL_NO_ERROR)
	{
		sprintf(textureErrorMessage,
				"glTexImage2D Error: %s\n", gluErrorString(errCode));
		error = true;
		return;
	}
}

// Opens and reads a .bmp file, which is a standard Microsoft bitmap
// format, output by the paint accessory.  LoadFromBMP allocates
// the memory buffer "textureBytes" to which its return value points.
void Texture::loadFromBMP (char*  filename)
{
    FILE* file;
    char  id[2];
    int   Size;
    char  Ignore1[4], Ignore2[100];
    int   OffBits;
    int   biSize;
    int   biWidth;
    int   biHeight;
    short biPlanes;
    short biBitCount;
    int   BMPlinelength;
    unsigned char* BMPline;
    int   startofline;
    GLubyte* texture1;
    int   newWidth, newHeight;
	bool  resize;
    int   retcode;
    const GLubyte *errString;
    int   additionalStuff;
    long  i,j;

    if ((file=fopen(filename,"rb"))==NULL)
    {
        sprintf(textureErrorMessage, "File Not Found: %s\n", filename);
        error = true;
		return;
    }

    // read "BM" identifier at start of file
    fread (id, 2, 1, file);
    if (id[0] != 'B' || id[1] != 'M')
    {
        sprintf(textureErrorMessage, "file %s doesn't have a BMP file header\n", 
				filename);
        error = true;
		return;
    }

    fread(&Size, 4, 1, file);
    fread(Ignore1, 4, 1, file);
    fread(&OffBits, 4, 1, file);
    fread(&biSize, 4, 1, file);
    fread(&biWidth, 4, 1, file);
    fread(&biHeight, 4, 1, file);
    fread(&biPlanes, 2, 1, file);
    fread(&biBitCount, 2, 1, file);
    additionalStuff = OffBits - 30;
    fread(Ignore2, 1, additionalStuff, file);

    printf("%s: %d x %d, %d bits per pixel\n",
           filename, biWidth, biHeight, biBitCount);

    if (biBitCount != 24)
    {
        sprintf(textureErrorMessage, "file %s isn't 24 bpp\n", filename);
        error = true;
		return;
    }

    textureBytes = (GLubyte*) malloc(biWidth*biHeight*4);

    // In .BMP files, the line width is padded to be a multiple of 4.
    j = (biWidth * 3) % 4;
    if (j > 0)
        j = 4-j;
    BMPlinelength = biWidth * 3 + j;
    BMPline = (unsigned char*) malloc(BMPlinelength);

    for (i=1; i<=biHeight; i++)
    {
        fread(BMPline, 1, BMPlinelength, file);
        startofline = (i - 1) * biWidth * 4;
        for (j=0; j<biWidth; j++)
        {
            textureBytes[startofline + (j*4)]   = BMPline[(j*3)+2];
            textureBytes[startofline + (j*4)+1] = BMPline[(j*3)+1];
            textureBytes[startofline + (j*4)+2] = BMPline[(j*3)];
            textureBytes[startofline + (j*4)+3] = 0xFF;
        }
    }

    // Make sure width and height are powers of two (borders
    // are not taken into account), which is required of textures.
    if (biWidth == 2 || biWidth == 4 || biWidth == 8 || biWidth == 16 ||
        biWidth == 32 || biWidth == 64 || biWidth == 128 || biWidth == 256 ||
        biWidth == 512 || biWidth == 1024)
		resize = false;
    else
    {
        resize = true;
        if (biWidth < 4) newWidth = 4;
        else if (biWidth < 8) newWidth = 8;
        else if (biWidth < 16) newWidth = 16;
        else if (biWidth < 32) newWidth = 32;
        else if (biWidth < 64) newWidth = 64;
        else if (biWidth < 128) newWidth = 128;
        else if (biWidth < 256) newWidth = 256;
        else if (biWidth < 512) newWidth = 512;
        else if (biWidth < 1024) newWidth = 1024;
        else
        {
            printf("biWidth of %d cannot be stretched, stopping\n",
                   biWidth);
            exit(1);
        }
    }
    if (biHeight == 2 || biHeight == 4 || biHeight == 8 || biHeight == 16 ||
        biHeight == 32 || biHeight == 64 || biHeight == 128 || biHeight == 256 || 
        biHeight == 512 || biHeight == 1024)
        ;
    else
    {
        resize = true;
        if (biHeight < 4) newHeight = 4;
        else if (biHeight < 8) newHeight = 8;
        else if (biHeight < 16) newHeight = 16;
        else if (biHeight < 32) newHeight = 32;
        else if (biHeight < 64)  newHeight = 64;
        else if (biHeight < 128) newHeight = 128;
        else if (biHeight < 256) newHeight = 256;
        else if (biHeight < 512) newHeight = 512;
        else if (biHeight < 1024) newHeight = 1024;
        else
        {
            sprintf(textureErrorMessage, 
				    "biHeight of %d cannot be stretched, stopping\n",
                    biHeight);
			error = true;
			return;
        }
    }
    if (resize)
    {
        texture1 = (GLubyte*) malloc(newWidth*newHeight*4);
        retcode = gluScaleImage(GL_RGBA, biWidth, biHeight, GL_UNSIGNED_BYTE,
                                textureBytes,
                                newWidth, newHeight, GL_UNSIGNED_BYTE,
                                texture1);
        if (retcode)
        {
            errString = gluErrorString(retcode);
            sprintf(textureErrorMessage, "gluScaleImage error: %s\n", 
				    errString);
			error = true;
			return;
        }
        free(textureBytes);
        textureBytes = texture1;
        width = newWidth;
        height = newHeight;
    }
    else
    {
        width = biWidth;
        height = biHeight;
    }

    fclose (file);
    free(BMPline);
}

void Texture::setParameter(GLenum pname, GLint param)
{
	switch (pname)
	{
	case GL_TEXTURE_WRAP_S:
		ParmWrapS = param;
		break;
	case GL_TEXTURE_WRAP_T:
		ParmWrapT = param;
		break;
	case GL_TEXTURE_MAG_FILTER:
		ParmMagFilter = param;
		break;
	case GL_TEXTURE_MIN_FILTER:
		ParmMinFilter = param;
		break;
	default:
        sprintf(textureErrorMessage, 
				"Unrecognized parameter %d for SetParameter\n", 
				pname);
		error = true;
		return;
	}
}

void Texture::makeCurrent()
{
    GLenum errCode;
	glBindTexture(GL_TEXTURE_2D, textureName);
	if ((errCode = glGetError()) != GL_NO_ERROR)
	{
		sprintf(textureErrorMessage,
				"glBindTexture Error: %s\n", gluErrorString(errCode));
		error = true;
		return;
	}

	if (ParmWrapS != 0)
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, ParmWrapS);
	if (ParmWrapT != 0)
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, ParmWrapT);
	if (ParmMagFilter != 0)
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, ParmMagFilter);
	if (ParmMinFilter != 0)
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, ParmMinFilter);
	if ((errCode = glGetError()) != GL_NO_ERROR)
	{
		sprintf(textureErrorMessage,
				"glTexParameteri Error: %s\n", gluErrorString(errCode));
		error = true;
		return;
	}
}

void Texture::makeNotCurrent()
{
	// Tell OpenGL to stop using texture objects
	glBindTexture(GL_TEXTURE_2D, 0);
}
